import { error } from '@sveltejs/kit';
import { PUBLIC_PRODUCTION_API_ORIGIN, PUBLIC_DEVELOPMENT_API_ORIGIN } from '$env/static/public';

/** @type {import('./$types').PageLoad} */
export async function load({ params, url, fetch, depends }) {
	// Add a dependency on the latest news endpoint
	depends('latest-news');

	const apiOrigin = import.meta.env.DEV
		? PUBLIC_DEVELOPMENT_API_ORIGIN
		: PUBLIC_PRODUCTION_API_ORIGIN;

	const { lang } = params;
	// Get page from URL query parameter or default to 1
	const page = parseInt(url.searchParams.get('page') || '1', 10);

	try {
		// Fetch initial posts
		const initialPostsResponse = await fetch(`${apiOrigin}/media/latest-news/${lang}`, {
			headers: {
				Accept: 'application/json'
			}
		});

		if (!initialPostsResponse.ok) {
			if (initialPostsResponse.status === 404) {
				throw error(404, {
					message: 'News not found'
				});
			}
			throw error(initialPostsResponse.status, {
				message: `Error: ${initialPostsResponse.statusText}`
			});
		}

		const initialData = await initialPostsResponse.json();

		// Check if response is our expected format (object with posts array)
		// or the old format (just array of posts)
		let initialPosts = [];
		let hasMore = false;

		if (initialData && typeof initialData === 'object' && initialData.posts) {
			// New API format with separate posts array and hasMore flag
			initialPosts = initialData.posts;
			hasMore = initialData.hasMore === true;
		} else if (Array.isArray(initialData)) {
			// Old API format (just an array of posts)
			initialPosts = initialData;
			hasMore = initialData.length >= 6; // If we got 6 initial posts, there might be more
		}

		if (!initialPosts || initialPosts.length === 0) {
			return {
				post: [],
				currentPage: page,
				hasMore: false
			};
		}

		// If page is greater than 1, fetch additional pages
		let allPosts = [...initialPosts];

		if (page > 1) {
			try {
				// Fetch all pages up to the requested page
				for (let i = 1; i < page; i++) {
					const morePostsResponse = await fetch(
						`${apiOrigin}/media/latest-news/${lang}/load-more/${i}`,
						{
							headers: {
								Accept: 'application/json'
							}
						}
					);

					if (!morePostsResponse.ok) {
						break;
					}

					const morePostsData = await morePostsResponse.json();

					if (morePostsData.posts && morePostsData.posts.length > 0) {
						allPosts = [...allPosts, ...morePostsData.posts];
						hasMore = morePostsData.hasMore === true;
					} else {
						hasMore = false;
						break;
					}
				}
			} catch (err) {
				console.error('Error loading additional pages:', err);
			}
		}

		return {
			post: allPosts,
			currentPage: page,
			hasMore
		};
	} catch (e) {
		console.error('Load error:', e);
		// Return empty posts array instead of throwing error to allow the page to render with fallback content
		return {
			post: [],
			currentPage: page,
			hasMore: false
		};
	}
}
